

rm(list=ls())

###################################################################################################################
####################### READ AND PREPARE GENERAL VARS #############################################################
###################################################################################################################

source("Packages.R")


dat = read.dta13("data_january2020.dta",
                 convert.factors = FALSE)
dat = zap_label(dat)
names(dat) = tolower(names(dat))


dat$age = dat$v_1
dat$female = as.numeric(dat$b == 2)
dat$urban = as.numeric(dat$c == 2)
dat$edu = car::recode(dat$v_50, "1=1; 2=1; 3=1; 4=1; 5=1; 6=1; 7=2; 8=2; 9=2; 10=3")
dat$income = dat$v_52

dat$ethnicity = dat$v_2
dat$javanese = car::recode(dat$ethnicity, "1=1; 2:13=0")
dat$sundanese = car::recode(dat$ethnicity, "1=0; 2=1; 3:13=0")

dat$javaisland = car::recode(dat$i1, "11=1; 12=1; 13=1; 14=1; 15=1; 16=1; else=0")

dat$religion = car::recode(dat$v_3, "1='Islam'; 2='Protestan'; 3='Katolik'; 4='Hindu'; 5='Buddha'; 6='Konghucu'; 7='Lainnya'")
dat$religion = as.factor(dat$religion)
dat$muslim = as.numeric(dat$religion == "Islam")

# Feeling: ethnic Chinese
dat$feelchinese = car::recode(dat$v_9_a, "8=NA")
# Feeling: Christian
dat$feelchristian = car::recode(dat$v_9_b, "8=NA")
# Feeling: Muslim
dat$feelmuslim = car::recode(dat$v_9_c, "8=NA")
# Feeling: Communists
  dat$feelcommunist = car::recode(dat$v_9_d, "8=NA")


# Capres 2019
dat$jokowi2019 = car::recode(dat$v_25, "2=0; 8=NA; 9=NA")


###################################################################################################################
## PREPARE EXPERIMENT SPECIFIC DATA
###################################################################################################################

dat$exp_2 = as.factor(dat$exp_2)
dat$group[dat$exp_2 == 1] = "Control"
dat$group[dat$exp_2 == 2] = "Natuna"
dat$group[dat$exp_2 == 3] = "Uighurs"
dat$group[dat$exp_2 == 4] = "Investment"
dat$group = factor(dat$group, levels=c("Control", "Natuna", "Uighurs", "Investment"))

## dependent variable: negative traits
dat$loyalty = car::recode(dat$v_17_a, "8=NA")
dat$selfinterested = car::recode(dat$v_17_b, "8=NA")
dat$culture = car::recode(dat$v_17_c, "8=NA")
dat$greedy = car::recode(dat$v_17_d, "8=NA")
dat$influence = car::recode(dat$v_17_e, "8=NA")

dat$negtraits = rowMeans(dat[, c("loyalty", "selfinterested", "culture", "greedy", "influence")],
                         na.rm=TRUE)

## dependent variable: uncomfortable with political leadership
dat$leadership = car::recode(dat$v_18, "1=4; 2=3; 3=2; 4=1; 8=NA")

## whether beneficial
dat$beneficial[dat$exp_2 == 1] = dat$v_16_a[dat$exp_2 == 1]
dat$beneficial[dat$exp_2 == 2] = dat$v_16_b[dat$exp_2 == 2]
dat$beneficial[dat$exp_2 == 3] = dat$v_16_c[dat$exp_2 == 3]
dat$beneficial[dat$exp_2 == 4] = dat$v_16_d[dat$exp_2 == 4]

dat$beneficial = car::recode(dat$beneficial, "8=NA")

cor(dat[, c("feelchinese", "negtraits", "leadership", "beneficial")], 
    use="pairwise.complete.obs")


## 3-level feeling chinese
dat$feelchinese_3lvl = car::recode(dat$feelchinese, "1='Negative'; 2='Negative'; 3='Neutral'; 4='Positive'; 5='Positive'")
dat$feelchinese_3lvl = factor(dat$feelchinese_3lvl, levels = c("Neutral", "Negative", "Positive"))


###################################################################################################################
## SURVEY
###################################################################################################################

svydat = svydesign(id=~1, data=dat, weights = dat$weight)
svydat.muslim = subset(svydat, subset=(muslim==1))
dat.muslim = subset(dat, subset=(muslim==1))

### PCA negative traits
negtraits.pca = svyprcomp(~loyalty+selfinterested+culture+greedy+influence, svydat.muslim)
summary(negtraits.pca)
negtraits.pca
psych::alpha(dat.muslim[, c("loyalty", "selfinterested", "culture", "greedy", "influence")])

###################################################################################################################
## REGRESSIONS
###################################################################################################################

## beneficial
lm.beneficial = svyglm(beneficial ~ group, svydat.muslim, family = "gaussian")
summary(lm.beneficial)

## negative traits
lm.negtraits = svyglm(negtraits ~ group, design = svydat.muslim, family = "gaussian")
summary(lm.negtraits)

## leadership
lm.leadership = svyglm(leadership ~ group, design = svydat.muslim, family = "gaussian")
summary(lm.leadership)


###################################################################################################################
## PLOT REGRESSION COEFFICIENTS
###################################################################################################################


plot.beneficial = 
  plot_summs(lm.beneficial,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Perceived Benefits of\na Close Tie with China") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        panel.border = element_rect(fill=NA, size=0.5)) +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.4, .4)) 
  


plot.negtraits = 
  plot_summs(lm.negtraits,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Negative Stereotypes\ntoward Ethnic Chinese") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        axis.text.y = element_blank(),
        panel.border = element_rect(fill=NA, size=0.5),
        plot.margin = unit(c(0,1,0,1),"cm"))  +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.4, .4)) 



plot.leadership = 
  plot_summs(lm.leadership,
             coefs=c("Territorial\nViolation"="groupNatuna",
                     "Uighurs\nDiscrimination" = "groupUighurs",
                     "Foreign\nInvestment"="groupInvestment"),
             colors = "black",
             inner_ci_level = .90) +
  ggtitle("Discomfort with Ethnic Chinese\nin Position of Political Power") +
  theme(plot.title=element_text(size=10, face="bold", hjust = 0.5),
        axis.text.y = element_blank(),
        panel.border = element_rect(fill=NA, size=0.5)) +
  geom_vline(xintercept = 0, color="red", size=0.4) +
  coord_cartesian(xlim=c(-.4, .4)) 



png("Figure 1a.png", width=3000, height=2000, res=300)
plot.beneficial + plot.negtraits + plot.leadership +
  plot_annotation(title = 'Marginal Effects of the Treatments (January 2020 Survey)')
dev.off()


###################################################################################################################
## ROBUSTNESS: INCLUDE NON-MUSLIMS
###################################################################################################################

### Include Non-Muslims
svylm.beneficial.nonmuslim = svyglm(beneficial ~ group, 
                                    design = svydat, family = "gaussian")
summary(svylm.beneficial.nonmuslim)

svylm.negtraits.nonmuslim = svyglm(negtraits ~ group, 
                                   design = svydat, family = "gaussian")
summary(svylm.negtraits.nonmuslim)

svylm.leadership.nonmuslim = svyglm(leadership ~ group, 
                                    design = svydat, family = "gaussian")
summary(svylm.leadership.nonmuslim)


###################################################################################################################
## ROBUSTNESS: ETHNIC CHINESE BARPLOT AND LINEAR
###################################################################################################################

### Barplot of Feeling of Ethnic Chinese
temp = data.frame(round(prop.table(table(dat.muslim$feelchinese))*100, 2))
colnames(temp) = c("Feeling", "Percentage")
temp$Feeling = c("Strong Dislike", "Dislike", "Neutral", "Like", "Strong Liking")
temp$Feeling = factor(temp$Feeling, 
                      c("Strong Dislike", "Dislike", "Neutral", "Like", "Strong Liking"))

plot_feelchinese = ggplot(data=temp, aes(x=Feeling, y=Percentage)) +
  geom_bar(stat="identity")+
  geom_text(aes(label=Percentage), vjust=-0.3, size=3.5)+
  theme_minimal() + 
  theme(axis.title.x = element_blank(),
        plot.title=element_text(size=10)) + 
  coord_cartesian(ylim=c(0, 70)) +
  ggtitle("Distribution of Feeling toward Ethnic Chinese, January 2020")

png("Figure 2a.png", width=2000, height=1000, res=300)
plot_feelchinese
dev.off()



###################################################################################################################
## ROBUSTNESS: ETHNIC CHINESE CATEGORICAL
###################################################################################################################

plot.int2 <- function(plot.main, title, obj, obj.label, pos.legend, legend.title.size, legend.label.size, lab.y, ylim, size.x=15, size.y=15) {
  plot.out <-
    plot.main +
    geom_errorbar(aes(ymin=Estimate - 1.96*se, ymax=Estimate + 1.96*se, col=obj), 
                  width=.1, lwd=1, position = position_dodge(width = .3)) +
    geom_errorbar(aes(ymin=Estimate - 1.64*se, ymax=Estimate + 1.64*se, col=obj), 
                  width=.1, lwd=2, position = position_dodge(width = .3)) +
    geom_point(aes(col=obj, pch=obj), 
               size=4, fill="black", position = position_dodge(width = .3)) +
    theme_bw() + 
    ylab(lab.y) +
    xlab("") +
    theme(axis.title.x = element_text(size = size.x)) +
    theme(axis.title.y = element_text(size = size.y)) +
    theme(axis.text.x = element_text(size = size.x)) +
    theme(axis.text.y = element_text(size = size.y)) +
    coord_cartesian(ylim=c(ylim[1], ylim[2])) + 
    theme(legend.text=element_text(size=legend.label.size)) + 
    guides(pch=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE), 
           col=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE)) +
    theme(legend.position = pos.legend) + 
    theme(legend.background = element_rect(fill="white", 
                                           size=.1, linetype="solid")) + 
    theme(plot.title = element_text(size = 10)) + 
    ggtitle(title)
  
  return (plot.out)
}

### Create Data
temp_beneficial = dat.muslim[complete.cases(dat.muslim[, c("group", "beneficial", "feelchinese_3lvl")]), ]
svytemp_beneficial = svydesign(id = ~1, data = temp_beneficial, weights = temp_beneficial$weight)

temp_negtraits = dat.muslim[complete.cases(dat.muslim[, c("group", "negtraits", "feelchinese_3lvl")]), ]
svytemp_negtraits = svydesign(id = ~1, data = temp_negtraits, weights = temp_negtraits$weight)

temp_leadership = dat.muslim[complete.cases(dat.muslim[, c("group", "beneficial", "feelchinese_3lvl")]), ]
svytemp_leadership = svydesign(id = ~1, data = temp_leadership, weights = temp_leadership$weight)


### Regression Benefits
int.beneficial.chinese.cat = svyglm(beneficial ~ group*feelchinese_3lvl, 
                                    design = svytemp_beneficial, family = "gaussian")
summary(int.beneficial.chinese.cat)

out_beneficial = summary(margins(int.beneficial.chinese.cat, design = svytemp_beneficial, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_beneficial = out_beneficial[out_beneficial$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_beneficial) = c("Treatment", "Feeling", "Estimate", "se")
out_beneficial$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                             "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                             "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_beneficial$Treatment = factor(out_beneficial$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_beneficial$Feeling = factor(out_beneficial$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_beneficial, aes(Treatment, Estimate))
plot_beneficial = plot.int2(plot.temp, 
                            title="Perceived Benefits",
                            obj=out_beneficial$Feeling,
                            obj.label = "Feeling toward Ethnic Chinese",
                            pos.legend = c(.87, .1),
                            legend.title.size = 9,
                            legend.label.size = 9,
                            lab.y = "Estimate",
                            ylim = c(-.9, .2),
                            size.x = 10,
                            size.y = 10) +
                            geom_hline(yintercept=0, linetype="dashed") +
                            coord_flip()


### Regression Negative Traits
int.negtraits.chinese.cat = svyglm(negtraits ~ group*feelchinese_3lvl, 
                                    design = svytemp_negtraits, family = "gaussian")
summary(int.negtraits.chinese.cat)

out_negtraits = summary(margins(int.negtraits.chinese.cat, design = svytemp_negtraits, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_negtraits = out_negtraits[out_negtraits$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_negtraits) = c("Treatment", "Feeling", "Estimate", "se")
out_negtraits$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                             "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                             "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_negtraits$Treatment = factor(out_negtraits$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_negtraits$Feeling = factor(out_negtraits$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_negtraits, aes(Treatment, Estimate))
plot_negtraits = plot.int2(plot.temp, 
                            title="Negative Stereotypes",
                            obj=out_negtraits$Feeling,
                            obj.label = "Feeling toward Ethnic Chinese",
                            pos.legend = c(.87, .1),
                            legend.title.size = 9,
                            legend.label.size = 9,
                            lab.y = "Estimate",
                            ylim = c(-.9, .2),
                            size.x = 10,
                            size.y = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()





### Regression Discomfort with Leadership
int.leadership.chinese.cat = svyglm(leadership ~ group*feelchinese_3lvl, 
                                   design = svytemp_leadership, family = "gaussian")
summary(int.leadership.chinese.cat)

out_leadership = summary(margins(int.leadership.chinese.cat, design = svytemp_leadership, at = list(feelchinese_3lvl = c("Neutral", "Negative", "Positive"))))
out_leadership = out_leadership[out_leadership$factor %in% c("groupInvestment", "groupNatuna", "groupUighurs"), c("factor", "feelchinese_3lvl", "AME", "SE")]

colnames(out_leadership) = c("Treatment", "Feeling", "Estimate", "se")
out_leadership$Treatment = c("Foreign\nInvestment","Foreign\nInvestment","Foreign\nInvestment",
                            "Territorial\nViolation","Territorial\nViolation","Territorial\nViolation",
                            "Uighurs\nDiscrimination","Uighurs\nDiscrimination","Uighurs\nDiscrimination")
out_leadership$Treatment = factor(out_leadership$Treatment, levels = c("Foreign\nInvestment", "Uighurs\nDiscrimination", "Territorial\nViolation"))
out_leadership$Feeling = factor(out_leadership$Feeling, levels = c("Neutral", "Negative", "Positive"))

plot.temp = ggplot(out_leadership, aes(Treatment, Estimate))
plot_leadership = plot.int2(plot.temp, 
                           title="Discomfort with Leadership",
                           obj=out_leadership$Feeling,
                           obj.label = "Feeling toward Ethnic Chinese",
                           pos.legend = c(.87, .1),
                           legend.title.size = 9,
                           legend.label.size = 9,
                           lab.y = "Estimate",
                           ylim = c(-.9, .2),
                           size.x = 10,
                           size.y = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()


png(filename = "Figure 3a.png", width=3000, height=2000, res=300)
(plot_beneficial + plot_negtraits + plot_leadership) +
  plot_annotation(
    title = 'Marginal Effects by Feeling toward Ethnic Chinese, January 2020') +
  plot_layout(guides = "collect") & theme(legend.position = "bottom")
dev.off()


###################################################################################################################
## OUTPUT REGRESSION MODELS
###################################################################################################################

tab_model(lm.beneficial,
          lm.negtraits,
          lm.leadership,
          
          show.se = TRUE, collapse.se = TRUE,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE,
          file="Regression January 2020 (Main).htm")


tab_model(int.beneficial.chinese.cat,
          int.negtraits.chinese.cat,
          int.leadership.chinese.cat,         
          
          svylm.beneficial.nonmuslim,
          svylm.negtraits.nonmuslim,
          svylm.leadership.nonmuslim,
          
          show.se = TRUE, collapse.se = TRUE,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE,
          file="Regression January 2020 (Robustness).htm")


###################################################################################################################
## COVARIATES BALANCE
###################################################################################################################

sink("Balance Check January 2020.txt")
# female
print("Female ==========")
summary(aov(female ~ group, subset(dat, muslim==1)))

# age
print("Age ==========")
summary(aov(age ~ group, subset(dat, muslim==1)))

# edu
print("Education ==========")
summary(aov(edu ~ group, subset(dat, muslim==1)))

# income
print("Income ==========")
summary(aov(income ~ group, subset(dat, muslim==1)))

# urban/rural
print("Urban/Rural ==========")
summary(aov(urban ~ group, subset(dat, muslim==1)))

# vote for jokowi
print("Jokowi 2019 ==========")
summary(aov(jokowi2019 ~ group, subset(dat, muslim==1)))

# feeling: chinese
print("Feeling Chinese ==========")
summary(aov(feelchinese ~ group, subset(dat, muslim==1)))

sink(NULL)